# ################################################################################### #
# Tcl script to report design review                                                  #
#                                                                                     #
#                                                                                     #
# Command arguments:                                                                  #
#                                                                                     #
#  -file filename: Specify report file name. If not included then no report file will #
#         generated, just a report to the tcl console.                                #
#                                                                                     #
#  -force: overwrite existing report file. Will be ignored if -file argument          #
#          not used.                                                                  #
#                                                                                     #
#  -run: name of run to be reported                                                   #
#                                                                                     #
# ################################################################################### #


proc report_design_review {args} {

   # set defaults for command line arguments
   set overWrite 0
   set reportFile ""
   set fileOutput 0
   set runName ""
     
   
   # parse command line arguments
   foreach arg $args {   
     if {[regexp -nocase -- {-force} $arg]} {
      set overWrite 1
     } elseif {[regexp -nocase -- {-file} $arg] == 1} {
	   regexp -nocase -- {-file\s+(\S+)} $args var1 reportFile
	   set fileOutput 1
     } elseif {[regexp -nocase -- {-run} $arg] == 1} {
	   regexp -nocase -- {-run\s+(\S+)} $args var1 runName
    }   
  }   

  set synthRunsList [get_runs -quiet -filter {IS_SYNTHESIS == true && PROGRESS == "100%"}]
  set impRunsList [get_runs -quiet -filter {IS_IMPLEMENTATION == true && PROGRESS == "100%"}]

  # Check if any completed runs exist
  if {[llength $impRunsList] == 0 && [llength $synthRunsList] == 0} {
      puts "ERROR: Couldn't find a completed run in this project."
      return -code error
   }

  if {$fileOutput == 1} {
     # check if destination file already exists
     # if yes and overwrite option, open in 'w' mode
     # if yes and no overwrite option, exit with warning
     # else, create a new file in 'w' mode
     if {[file exists $reportFile] && ($overWrite != 1)} {
      puts "ERROR: Output file exists already, provide new output file name or set overwrite option (-force)."
      return -code error
     } elseif {![file exists $reportFile]} {
       puts "Creating output file.."
     }
     set toFile [open $reportFile w+]	  
   }



   # check run exists in project and is completed
   set impRunIndex [lsearch $impRunsList $runName]
   set synthRunIndex [lsearch $synthRunsList $runName]

   if {$impRunIndex != -1} {
       set run [lindex $impRunsList $impRunIndex]
   } elseif {$synthRunIndex != -1} {
       set run [lindex $synthRunsList $synthRunIndex]
       if {$fileOutput == 1} {
         puts $toFile " ** WARNING: Checks run on synthesized design, not implemented design **"
         puts $toFile ""
       } 
       puts "WARNING: Checks run on synthesized design, not implemented design."
   } else {
      puts "ERROR: Couldn't find a completed run with that name."
      return -code error
   }



# open the run
open_run $run

set runName [get_property NAME [current_design]]
set part [get_property PART [current_design]]
set runVivVer   [ string range  [get_property MLO_VERSION_NUMBER [current_design]] 0 5 ]
set ver [ string range [version] 8 13 ]
set systemTime [clock seconds]
set timeNow [clock format $systemTime -format %H:%M:%S]
set dateNow [clock format $systemTime -format %a-%d-%b-%Y]


   if {$fileOutput == 1} {
  	
     puts "Writing results to $reportFile.."
     
     # print header to file
     puts $toFile "-------------------------------------------------------------------------------------"
     puts $toFile " DESIGN REVIEW REPORT"
     puts $toFile ""
     puts $toFile " PROJECT: [get_property DIRECTORY [current_project]]/[get_property NAME [current_project]].xpr"
     puts $toFile ""   
     puts $toFile " RUN NAME: $runName"
     puts $toFile ""   
     puts $toFile " PART: $part"
     puts $toFile ""   
     puts $toFile " DESIGN RUN GENERATED WITH: Vivado $runVivVer"
     puts $toFile ""   
     puts $toFile " REPORT GENERATED WITH: Vivado $ver"
     puts $toFile ""
     puts $toFile " GENERATED: $dateNow $timeNow"
     puts $toFile "-------------------------------------------------------------------------------------"
     puts $toFile "" 
   }
   
 
   # print header to console
   puts "-------------------------------------------------------------------------------------"
   puts " DESIGN REVIEW REPORT"
   puts ""
   puts " PROJECT: [get_property DIRECTORY [current_project]]/[get_property NAME [current_project]].xpr"
   puts ""   
   puts " RUN NAME: $runName"
   puts ""
   puts " PART: $part"
   puts ""   
   puts " DESIGN RUN GENERATED WITH: Vivado $runVivVer"
   puts ""   
   puts " REPORT GENERATED WITH: Vivado $ver"
   puts ""
   puts " GENERATED: $dateNow $timeNow"
   puts "-------------------------------------------------------------------------------------"
   puts "" 
 



####################################
#  DESIGN CORRECTNESS
####################################
# DRC checks

if {$fileOutput == 1} {
  puts $toFile "-------------------------------------------------------------------------------------"
  puts $toFile " DESIGN CORRECTNESS"
  puts $toFile "-------------------------------------------------------------------------------------"
  puts $toFile "" 
}
puts "-------------------------------------------------------------------------------------"
puts " DESIGN CORRECTNESS"
puts "-------------------------------------------------------------------------------------"
puts "" 



 set checkResults [report_drc -return_string -ruledecks {default opt_checks placer_checks router_checks bitstream_checks}]
 set searchStart [string first "2. REPORT DETAILS" $checkResults]
 set startIndex [string first "Violations" $checkResults $searchStart]
 set endIndex [string first "2. REPORT DETAILS" $checkResults $searchStart+1]
 set filtered  [string range $checkResults $startIndex $endIndex-1]


 if {$fileOutput == 1} {
   puts $toFile "REPORT DRC:"
   puts $toFile $filtered
 }
 puts "REPORT DRC:"
 puts $filtered



# Methodology checks - no XDC or Timing checks
 set checkResults [report_methodology -return_string -checks {PDRC-204 PDRC-190 CLKC-54 CLKC-53 CLKC-9 CLKC-5 CLKC-38 CLKC-37 CLKC-36 CLKC-35 \
CLKC-4 CLKC-18 CLKC-3 CLKC-2 CLKC-1 REQP-1959 LUTAR-1 HPDR-1 NTCN-1 CKLD-2 CKLD-1 DPIR-1 CKBF-1 SYNTH-14 SYNTH-13 SYNTH-12 SYNTH-11 SYNTH-10 SYNTH-9 SYNTH-16 SYNTH-15 SYNTH-6 SYNTH-5 SYNTH-4}]
 set searchStart [string first "2. REPORT DETAILS" $checkResults]
 set startIndex [string first "Violations" $checkResults $searchStart]
 set endIndex [string first "2. REPORT DETAILS" $checkResults $searchStart+1]
 set filtered  [string range $checkResults $startIndex $endIndex-1]




if {$fileOutput == 1} { 
  puts $toFile "REPORT METHODOLOGY:"
  puts $toFile $filtered
}
 puts "REPORT METHODOLOGY:"
puts $filtered




####################################
#  CONSTRAINTS VALIDATION
####################################

if {$fileOutput == 1} {
puts $toFile "-------------------------------------------------------------------------------------"
puts $toFile " CONSTRAINTS VALIDATION"
puts $toFile "-------------------------------------------------------------------------------------"
puts $toFile "" 
}
puts "-------------------------------------------------------------------------------------"
puts " CONSTRAINTS VALIDATION"
puts "-------------------------------------------------------------------------------------"
puts "" 


# Methodology checks - only XDC or Timing checks
 set checkResults [report_methodology -return_string -checks {TIMING-43 TIMING-42 TIMING-41 TIMING-40 TIMING-39 TIMING-38 TIMING-37 TIMING-36 \
TIMING-35 TIMING-34 TIMING-33 TIMING-32 TIMING-31 TIMING-30 TIMING-29 TIMING-28 TIMING-27 TIMING-26 TIMING-25 TIMING-24 TIMING-23 TIMING-22 \
TIMING-21 TIMING-20 TIMING-19 TIMING-18 TIMING-17 TIMING-16 TIMING-15 TIMING-14 TIMING-13 TIMING-12 TIMING-11 TIMING-10 TIMING-9 TIMING-8 \
TIMING-7 TIMING-6 TIMING-5 TIMING-4 TIMING-3 TIMING-2 TIMING-1 XDCV-2 XDCV-1 XDCH-2 XDCH-1 XDCC-8 XDCC-7 XDCC-6 XDCC-5 XDCC-4 XDCC-3 XDCC-2 XDCC-1 XDCB-5 XDCB-4 XDCB-3 XDCB-2 XDCB-1}]

 set searchStart [string first "2. REPORT DETAILS" $checkResults]
 set startIndex [string first "Violations" $checkResults $searchStart]
 set endIndex [string first "2. REPORT DETAILS" $checkResults $searchStart+1]
 set filtered  [string range $checkResults $startIndex $endIndex-1]

if {$fileOutput == 1} { 
  puts $toFile "REPORT METHODOLOGY:"
  puts $toFile $filtered
}
 puts "REPORT METHODOLOGY:"
puts $filtered


 set checkResults [check_timing -verbose -return_string]
 set searchStart [string first "1. checking no_clock" $checkResults]
 set startIndex [string first "1. checking no_clock" $checkResults $searchStart+1]
 set filtered  [string range $checkResults $startIndex end]


if {$fileOutput == 1} { 
  puts $toFile "CHECK TIMING:"
  puts $toFile ""
  puts $toFile $filtered
}
puts "CHECK TIMING:"
puts ""
puts $filtered


 set checkResults [report_exceptions -no_header -ignored -return_string]

if {$fileOutput == 1} { 
  puts $toFile "IGNORED EXCEPTIONS:"
  puts $toFile ""
  puts $toFile $checkResults
}
puts "IGNORED EXCEPTIONS:"
puts ""
puts $checkResults



 ####################################
 #  CLOCK DOMAIN CROSSING
 ####################################
 
 if {$fileOutput == 1} {
 puts $toFile "-------------------------------------------------------------------------------------"
 puts $toFile " CLOCK DOMAIN CROSSING"
 puts $toFile "-------------------------------------------------------------------------------------"
 puts $toFile "" 
 }
 puts "-------------------------------------------------------------------------------------"
 puts " CLOCK DOMAIN CROSSING"
 puts "-------------------------------------------------------------------------------------"
 puts "" 
 
 
  set checkResults [report_cdc -return_string -no_header -summary]
   if {$fileOutput == 1} {  
    puts $toFile $checkResults
  }
   puts $checkResults
 
 

 ####################################
 #  CONSTRAINTS MET?
 ####################################
 
 if {$fileOutput == 1} {
 puts $toFile "-------------------------------------------------------------------------------------"
 puts $toFile " CONSTRAINTS MET?"
 puts $toFile "-------------------------------------------------------------------------------------"
 puts $toFile "" 
 }
 puts "-------------------------------------------------------------------------------------"
 puts " CONSTRAINTS MET?"
 puts "-------------------------------------------------------------------------------------"
 puts "" 
 
  set checkResults [report_timing -no_header -return_string -delay_type min_max -max_paths 1 -path_type summary -slack_lesser_than 0.0 -sort_by group -significant_digits 2]
   if {$fileOutput == 1} {  
    puts $toFile $checkResults
  }
   puts $checkResults

   

close_design

  if {$fileOutput == 1} {  
    close $toFile
  }
	
puts "---- Completed ----" 


}

